<?php

namespace App\Controllers;

class MercadoLibreOrders extends BaseController
{
    protected $configModel;
    protected $mlModel;
    protected $mlOrdersModel;

    public function __construct()
    {
        $this->configModel = new \App\Models\ConfigModel();
        $this->mlModel = new \App\Models\MercadoLibreModel();
        $this->mlOrdersModel = new \App\Models\MercadoLibreOrdersModel();
    }

    public function index()
    {
        return view('mercadolibre_orders');
    }

    /**
     * Listar órdenes para DataTables
     */
    public function listarOrdenes()
    {
        if ($this->request->isAJAX()) {
            $start = $this->request->getGet('start') ?? 0;
            $length = $this->request->getGet('length') ?? 10;
            $search = $this->request->getGet('search')['value'] ?? '';
            $order = $this->request->getGet('order')[0] ?? null;

            // Definir las columnas para ordenar
            $columns = ['order_id', 'date_created', 'buyer_nickname', 'items_count', 'total_amount', 'status'];
            $orderColumn = $columns[$order['column'] ?? 0] ?? 'date_created';
            $orderDir = $order['dir'] ?? 'DESC';

            $result = $this->mlOrdersModel->getOrdersPaginated(
                $start,
                $length,
                $search,
                $orderColumn,
                $orderDir
            );

            return $this->response->setJSON([
                'draw' => $this->request->getGet('draw'),
                'recordsTotal' => $result['recordsTotal'],
                'recordsFiltered' => $result['recordsFiltered'],
                'data' => $result['data']
            ]);
        }
    }

    /**
     * Sincronizar órdenes desde MercadoLibre
     */
    public function sincronizarOrdenes()
    {
        if (!$this->configModel->getConfig('ml_sync_active')['valor']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Sincronización desactivada'
            ]);
        }

        try {
            $accessToken = $this->refreshToken();
            $fechaActual = date('Y-m-d');
            $fechaUnMesAntes = date('Y-m-d', strtotime('-1 month'));
            
            $totalOrdenes = 0;
            $nuevasOrdenes = 0;
            $ordenesActualizadas = 0;
            
            // Configuración inicial para la paginación
            $offset = 0;
            $limit = 50;
            $hayMasResultados = true;
            
            while ($hayMasResultados) {
                // Construir la URL de la API
                $url = 'https://api.mercadolibre.com/orders/search?' .
                       'seller=' . $this->getUserId($accessToken) . '&' .
                       'limit=' . $limit . '&' .
                       'offset=' . $offset . '&' .
                       'order.status=paid&' .
                       'order.date_created.from=' . $fechaUnMesAntes . 'T00:00:00.000-00:00&' .
                       'order.date_created.to=' . $fechaActual . 'T23:59:59.000-00:00';
                
                // Configurar cURL
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Authorization: Bearer ' . $accessToken,
                    'Content-Type: application/json'
                ]);
                
                // Ejecutar la solicitud
                $response = curl_exec($ch);
                curl_close($ch);
                
                $responseData = json_decode($response, true);
                
                if (!isset($responseData['results']) || empty($responseData['results'])) {
                    $hayMasResultados = false;
                    continue;
                }
                
                // Procesar cada orden
                foreach ($responseData['results'] as $order) {
                    $totalOrdenes++;
                    
                    // Preparar los datos de la orden
                    $orderData = [
                        'order_id' => $order['id'],
                        'date_created' => date('Y-m-d H:i:s', strtotime($order['date_created'])),
                        'date_closed' => isset($order['date_closed']) ? date('Y-m-d H:i:s', strtotime($order['date_closed'])) : null,
                        'status' => $order['status'],
                        'total_amount' => $order['total_amount'],
                        'currency_id' => $order['currency_id'],
                        'buyer_id' => $order['buyer']['id'],
                        'buyer_nickname' => $order['buyer']['nickname'],
                        'shipping_id' => $order['shipping']['id'] ?? null,
                        'shipping_status' => $order['shipping']['status'] ?? null,
                        'items_count' => count($order['order_items']),
                        'payment_method' => $this->getPaymentMethod($order),
                        'order_json' => json_encode($order),
                        'fecha_registro' => date('Y-m-d H:i:s')
                    ];
                    
                    // Verificar si ya existe
                    $existeOrden = $this->mlOrdersModel->where('order_id', $orderData['order_id'])->first();
                    
                    if ($existeOrden) {
                        $this->mlOrdersModel->update($existeOrden['id'], $orderData);
                        $ordenesActualizadas++;
                    } else {
                        $this->mlOrdersModel->insert($orderData);
                        $nuevasOrdenes++;
                    }
                }
                
                // Verificar si hay más resultados para paginar
                if (count($responseData['results']) < $limit) {
                    $hayMasResultados = false;
                } else {
                    $offset += $limit;
                }
            }
            
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Sincronización completada',
                'total_ordenes' => $totalOrdenes,
                'nuevas_ordenes' => $nuevasOrdenes,
                'ordenes_actualizadas' => $ordenesActualizadas
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error al sincronizar órdenes: ' . $e->getMessage()
            ]);
        }
    }
    
    /**
     * Ver detalle de una orden
     */
    public function detalleOrden($orderId)
    {
        $orden = $this->mlOrdersModel->where('order_id', $orderId)->first();
        
        if (!$orden) {
            return redirect()->to(base_url('mercadolibre_orders'))->with('error', 'Orden no encontrada');
        }
        
        $data['orden'] = $orden;
        $data['ordenJson'] = json_decode($orden['order_json'], true);
        
        return view('ml_order_detail', $data);
    }
    
    /**
     * Crear factura desde una orden
     */
    // public function crearFactura()
    // {
    //     if ($this->request->isAJAX()) {
    //         $orderId = $this->request->getPost('order_id');
            
    //         try {
    //             // Obtener la orden
    //             $orden = $this->mlOrdersModel->where('order_id', $orderId)->first();
    //             if (!$orden) {
    //                 throw new \Exception('Orden no encontrada');
    //             }
                
    //             // Verificar si ya fue procesada
    //             if ($orden['processed'] == 1) {
    //                 throw new \Exception('Esta orden ya ha sido procesada. Factura: ' . $orden['invoice_id']);
    //             }
                
    //             $ordenData = json_decode($orden['order_json'], true);
                
    //             // Aquí implementarías la lógica para crear la factura en tu sistema
    //             // Por ejemplo, podrías tener un modelo de facturas que se encargue de esto
                
    //             // Simulando creación de factura (reemplazar con lógica real)
    //             $invoiceId = 'INV-' . time();
                
    //             // Marcar la orden como procesada
    //             $this->mlOrdersModel->markAsProcessed($orderId, $invoiceId);
                
    //             return $this->response->setJSON([
    //                 'success' => true,
    //                 'message' => 'Factura creada correctamente',
    //                 'invoice_id' => $invoiceId
    //             ]);
                
    //         } catch (\Exception $e) {
    //             return $this->response->setJSON([
    //                 'success' => false,
    //                 'message' => $e->getMessage()
    //             ]);
    //         }
    //     }
    // }
    public function crearFactura()
    {
        if ($this->request->isAJAX()) {
            $orderId = $this->request->getPost('order_id');
            
            try {
                // Obtener la orden
                $orden = $this->mlOrdersModel->where('order_id', $orderId)->first();
                if (!$orden) {
                    throw new \Exception('Orden no encontrada');
                }
                
                // Verificar si ya fue procesada
                if ($orden['processed'] == 1) {
                    throw new \Exception('Esta orden ya ha sido procesada. Factura: ' . $orden['invoice_id']);
                }
                
                // Decodificar los datos JSON de la orden
                $ordenData = json_decode($orden['order_json'], true);
                
                // Iniciar transacción
                $db = \Config\Database::connect();
                $db->transStart();
                
                try {
                    // Obtener el siguiente número de factura desde SACORRELSIS
                    $correlativo = $this->obtenerCorrelativo('PrxFact');
                    if (!$correlativo) {
                        throw new \Exception('Error al obtener el correlativo para la factura');
                    }
                    
                    // Buscar o crear cliente
                    $codCliente = $this->obtenerOCrearCliente($ordenData);
                    
                    // Fecha actual para la factura
                    $fechaActual = date('Y-m-d H:i:s');
                    
                    // 1. Crear encabezado de factura en SAFACT
                    $facturaData = [
                        'CodSucu' => '00000', // Ajustar según tu configuración
                        'TipoFac' => 'A', // Tipo de factura, ajustar según necesidad
                        'NumeroD' => $correlativo['NumeroD'],
                        //'NroUnico' => $correlativo['NroUnico'],
                        'FromTran' => 1, // Indicador que viene de MercadoLibre
                        'CodUsua' => 'MLAUTO', // Usuario que crea la factura
                        'Signo' => 1, // 1 para ventas, -1 para devoluciones
                        'FechaT' => date('m-d-Y H:i', strtotime($orden['date_created'])),
                        'FechaE' => date('m-d-Y H:i', strtotime($orden['date_created'])),
                        'FechaR' => date('m-d-Y H:i', strtotime($orden['date_created'])),
                        'FechaV' => date('m-d-Y H:i', strtotime($orden['date_created'])), // 30 días de vencimiento
                        'Moneda' => $orden['currency_id'],
                        'Factor' => 1, // Factor de conversión, ajustar si es necesario
                        'CodClie' => $codCliente,
                        'Descrip' => substr($ordenData['buyer']['nickname'] ?? 'Cliente ML', 0, 60),
                        'ID3' => $ordenData['buyer']['id'] ?? '',
                        'Monto' => $orden['total_amount'],
                        'MtoTotal' => $orden['total_amount'],
                        'Contado' => $orden['total_amount'], // Asumiendo que todas las ventas de ML son de contado
                        'Credito' => 0,
                        'Notas1' => 'Orden ML: ' . $orden['order_id']
                    ];
                    
                    // Calcular totales de impuestos y gravables
                    $totalesFact = $this->calcularTotalesFactura($ordenData['order_items'] ?? []);
                    $facturaData = array_merge($facturaData, $totalesFact);
                    
                    // Insertar en SAFACT
                    $db->table('SAFACT')->insert($facturaData);

                    $logger = \Config\Services::logger();
                    $logger->error('Este es un mensaje de error');
                    
                    // 2. Crear los detalles de la factura (SAITEMFAC)
                    $this->crearDetallesFactura($orden, $ordenData, $facturaData, $db);
                    
                    // 3. Marcar la orden como procesada
                    $this->mlOrdersModel->markAsProcessed($orderId, $facturaData['NumeroD']);
                    
                    // Confirmar transacción
                    $db->transComplete();
                    
                    if ($db->transStatus() === false) {
                        throw new \Exception('Error en la transacción de la base de datos');
                    }
                    
                    return $this->response->setJSON([
                        'success' => true,
                        'message' => 'Factura creada correctamente',
                        'invoice_id' => $facturaData['NumeroD']
                    ]);
                    
                } catch (\Exception $e) {
                    // Revertir la transacción en caso de error
                    $db->transRollback();
                    throw $e;
                }
                
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * Obtiene el siguiente correlativo para la factura
     */
    private function obtenerCorrelativo($tipoDoc)
    {
        $db = \Config\Database::connect();
        
        // Buscar el correlativo actual
        $correlativo = $db->table('SACORRELSIS')
            ->where('FieldName', $tipoDoc)
            // ->where('CodSucu', '01') // Ajustar según configuración
            ->get()
            ->getRowArray();
        
        if (!$correlativo) {
            return null;
        }
        
        // Obtener el siguiente número
        $nroUnico = intval($correlativo['ValueInt']) + 1;
        // $numeroD = $correlativo['Prefijo'] . str_pad($nroUnico, $correlativo['LenCorrel'], '0', STR_PAD_LEFT);
        
        // Actualizar el correlativo
        $db->table('SACORRELSIS')
            ->where('FieldName', $tipoDoc)
            ->update(['ValueInt' => $nroUnico]);
        
        return [
            'NroUnico' => $nroUnico,
            'NumeroD' => $nroUnico
        ];
    }

    /**
     * Busca o crea un cliente basado en la información de la orden
     */
    private function obtenerOCrearCliente($ordenData)
    {
        $db = \Config\Database::connect();
        
        // Obtener datos del comprador
        $buyerId = $ordenData['buyer']['id'] ?? '';
        $buyerNickname = $ordenData['buyer']['nickname'] ?? 'Cliente ML';
        
        // Verificar si el cliente ya existe
        $cliente = $db->table('SACLIE')
            ->where('ID3', (string)$buyerId)
            ->get()
            ->getRowArray();
        
        if ($cliente) {
            return $cliente['CodClie'];
        }
        
        // Si no existe, crear uno nuevo
        // Generar un código de cliente
        $maxCodCliente = $db->table('SACLIE')
            ->selectMax('CodClie')
            ->get()
            ->getRowArray();
        
        $nuevoCodCliente = 'ML' . str_pad((intval(substr($maxCodCliente['CodClie'] ?? 'ML0000', 2)) + 1), 4, '0', STR_PAD_LEFT);
        
        // Preparar datos del cliente
        $clienteData = [
            'CodClie' => $buyerId,
            'Descrip' => substr($buyerNickname, 0, 60),
            'ID3' => $buyerId,
            'TipoID3' => 1, // Ajustar según corresponda
            'Activo' => 1,
            'Email' => $ordenData['buyer']['email'] ?? '',
            'TipoCli' => 1,
            'FechaE' => date('Y-m-d H:i:s'),
            'TipoPVP' => 1, // Ajustar según tu configuración
            'EsCredito' => 0 // Las ventas de ML son de contado
        ];
        
        // Insertar el cliente
        $db->table('SACLIE')->insert($clienteData);
        
        return $nuevoCodCliente;
    }

    /**
     * Calcula los totales de la factura
     */
    private function calcularTotalesFactura($items)
    {
        $totalGravable = 0;
        $totalExento = 0;
        $totalImpuesto = 0;
        $totalCosto = 0;
        
        foreach ($items as $item) {
            $cantidad = $item['quantity'] ?? 1;
            $precioUnitario = $item['unit_price'] ?? 0;
            $subtotal = $cantidad * $precioUnitario;

            $sku = '';
            if (isset($item['item']['seller_custom_field'])) {
                $sku = $item['item']['seller_custom_field'];
            } elseif (isset($item['item']['seller_sku'])) {
                $sku = $item['item']['seller_sku'];
            }
            
            // Determinar si el producto tiene impuesto
            //$tieneImpuesto = $this->productoTieneImpuesto($item['item']['id'] ?? '');
            
            //if ($tieneImpuesto) {
                // Suponiendo un IVA del 16% (ajustar según corresponda)
                $impuesto = $subtotal * 0.19;
                $totalGravable += $subtotal;
                $totalImpuesto += $impuesto;
            //} else {
            //    $totalExento += $subtotal;
            //}
            
            // Obtener el costo del producto
            
            $costo = $this->obtenerCostoProducto($sku);
            $totalCosto += $costo * $cantidad;
        }
        
        return [
            'TGravable' => $totalGravable,
            'TExento' => $totalExento,
            'MtoTax' => $totalImpuesto,
            'CostoPrd' => $totalCosto
        ];
    }

    /**
     * Verifica si un producto tiene impuesto
     */
    private function productoTieneImpuesto($mlItemId)
    {
        // Implementar lógica para determinar si un producto tiene impuesto
        // Por defecto, asumimos que todos los productos tienen impuesto
        return true;
    }

    /**
     * Obtiene el costo de un producto
     */
    private function obtenerCostoProducto($sku)
    {
        $db = \Config\Database::connect();
        
        // Obtener el costo desde SAPROD
        $saproducto = $db->table('SAPROD')
            ->select('CostPro')
            ->where('Descrip2', $sku)
            ->get()
            ->getRowArray();
        
        return $saproducto['CostPro'] ?? 0;
    }

    /**
     * Crea los detalles de la factura
     */
    private function crearDetallesFactura($orden, $ordenData, $facturaData, $db)
    {
        $logger = \Config\Services::logger();
        $logger->error('Creacion de Detalles de factura');
        $logger->error(json_encode($ordenData));
        $items = $ordenData['order_items'];
        $nroLinea = 1;

        $logger->error('Datos de : '. json_encode($ordenData['order_items']));
        
        foreach ($items as $item) {
            // 1. Con el SKU de ML se obtiene el CODPROD

            $logger->error('INFO ITEM');

            $mlItemId = $item['item']['id'] ?? '';
            $sku = '';
            if (isset($item['item']['seller_custom_field'])) {
                $sku = $item['item']['seller_custom_field'];
            } elseif (isset($item['item']['seller_sku'])) {
                $sku = $item['item']['seller_sku'];
            }
            //$codProd = $this->obtenerCodProdDesdeML($mlItemId);
            
            //if (!$codProd) {
                // Si no se encuentra el producto, crear un registro de producto temporal
            //    $codProd = $this->crearProductoDesdeItemML($item);
            //}
            
            // 2. Obtener datos del producto desde SAPROD

            try{
                $producto = $db->table('SAPROD')
                ->where('Descrip2', $sku)
                ->get()
                ->getRowArray();

                $logger->error('Datos de : ');
                $logger->error('Este es un mensaje informativo: ');
                $logger->error('Este es un mensaje informativo: ');
                $logger->error('Datos de depuración: ' . json_encode($producto));

                if (!$producto) {
                    throw new \Exception('No se pudo obtener o crear el cliente');
                }

            }
            catch (\Exception $e) {
                // Revertir la transacción en caso de error
                $db->transRollback();
                throw $e;
            }
            
            //if (!$producto) {
            //    continue; // Saltar este item si no se encuentra el producto
            //}
            
            // 3 y 4. Obtener existencia anterior
            $existencia = $this->obtenerExistenciaAnterior($sku, '01'); // 01 es el código de ubicación/bodega
            
            // 5. Calcular precio neto del artículo
            $cantidad = $item['quantity'] ?? 1;
            $precioUnitario = $item['unit_price'] ?? 0;
            $subtotal = $cantidad * $precioUnitario;
            
            // 6. Calcular impuesto
            // $tieneImpuesto = $this->productoTieneImpuesto($mlItemId);
            $impuesto = ($subtotal * 0.16); // Suponiendo un IVA del 16%
            
            // 7. Preparar los datos del ítem de factura
            $itemFacturaData = [
                'CodSucu' => $facturaData['CodSucu'],
                'TipoFac' => $facturaData['TipoFac'],
                'NumeroD' => $facturaData['NumeroD'],
                'NroLinea' => $nroLinea,
                'CodItem' => $sku,
                'CodUbic' => '01', // Ajustar según corresponda
                'Descrip1' => $producto['Descrip'] ?? $item['item']['title'] ?? '',
                'Descrip2' => $producto['Descrip2'] ?? '',
                'Descrip3' => $producto['Descrip3'] ?? '',
                'Refere' => $producto['Refere'] ?? $mlItemId,
                'Signo' => 1,
                'Cantidad' => $cantidad,
                'ExistAnt' => $existencia,
                'Costo' => $subtotal,
                'Precio' => $precioUnitario,
                'MtoTax' => $impuesto,
                'TotalItem' => $subtotal + $impuesto,
                'FechaE' => date('Y-m-d H:i:s'), // 7. Fecha de la transacción
                'EsServ' => 0, // Asumimos que son productos, no servicios
                'EsExento' => 0
            ];
            
            try{

            
                // Insertar el ítem de factura
            $resultadoSaitemfac = $db->table('SAITEMFAC')->insert($itemFacturaData);

            $db->transComplete();

            if ($db->transStatus() === false) {
                throw new \Exception('Error en la transacción de la base de datos');
            }

            } catch (\Exception $e) {
                // Revertir la transacción en caso de error
                $db->transRollback();
                throw $e;
            }
            
            if (!$resultadoSaitemfac) {
                throw new \Exception('Error al insertar en SAITEMFACT: ' . $db->error()['message']);
            }
            
            
            
            // Actualizar el inventario (reducir existencia)
            $this->actualizarInventario($sku, '01', $cantidad);
            
            $nroLinea++;
        }
    }

    /**
     * Obtiene el CodProd desde el ítem de MercadoLibre
     */
    private function obtenerCodProdDesdeML($mlItemId)
    {
        $db = \Config\Database::connect();
        
        $producto = $db->table('ml_productos_sync')
            ->select('cod_prod')
            ->where('ml_id', $mlItemId)
            ->get()
            ->getRowArray();
        
        return $producto['cod_prod'] ?? null;
    }

    /**
     * Crea un producto temporal desde un ítem de MercadoLibre
     */
    private function crearProductoDesdeItemML($item)
    {
        $db = \Config\Database::connect();
        
        // Generar un código de producto temporal
        $maxCodProd = $db->table('SAPROD')
            ->selectMax('CodProd')
            ->like('CodProd', 'ML%')
            ->get()
            ->getRowArray();
        
        $nuevoCodProd = 'ML' . str_pad((intval(substr($maxCodProd['CodProd'] ?? 'ML0000', 2)) + 1), 4, '0', STR_PAD_LEFT);
        
        // Datos del producto
        $productoData = [
            'CodProd' => $nuevoCodProd,
            'Descrip' => substr($item['item']['title'] ?? 'Producto ML', 0, 60),
            'Activo' => 1,
            'Refere' => $item['item']['id'] ?? '',
            'Costo' => $item['unit_price'] * 0.7, // Costo estimado (70% del precio de venta)
            'Precio1' => $item['unit_price'] ?? 0
        ];
        
        // Insertar el producto
        $db->table('SAPROD')->insert($productoData);
        
        // Crear relación en ml_productos_sync
        $syncData = [
            'ml_id' => $item['item']['id'] ?? '',
            'cod_prod' => $nuevoCodProd,
            'ultimo_sync' => date('Y-m-d H:i:s')
        ];
        
        $db->table('ml_productos_sync')->insert($syncData);
        
        return $nuevoCodProd;
    }

    /**
     * Obtiene la existencia anterior de un producto
     */
    private function obtenerExistenciaAnterior($codProd, $codUbic)
    {
        $db = \Config\Database::connect();

        $logger = \Config\Services::logger();
        
        $existencia = $db->table('SAEXIS')
            ->select('Existen')
            ->where('CodProd', $codProd)
            ->where('CodUbic', $codUbic)
            ->get()
            ->getRowArray();

        $logger->info('Datos de Existen: ');
        $logger->info('Este es un mensaje informativo: ');

        $logger->info('Datos de depuración: ' . json_encode($existencia));
        
        return $existencia['Existen'] ?? 0;
    }

    /**
     * Actualiza el inventario tras una venta
     */
    private function actualizarInventario($codProd, $codUbic, $cantidad)
    {
        $db = \Config\Database::connect();
        
        // Verificar si existe el registro en SAEXIS
        $existencia = $db->table('SAEXIS')
            ->where('CodProd', $codProd)
            ->where('CodUbic', $codUbic)
            ->get()
            ->getRowArray();
        
        if ($existencia) {
            // Actualizar la existencia
            $nuevaExistencia = $existencia['Existen'] - $cantidad;
            
            $db->table('SAEXIS')
                ->where('CodProd', $codProd)
                ->where('CodUbic', $codUbic)
                ->update(['Existen' => $nuevaExistencia]);
        } else {
            // Crear un nuevo registro
            $db->table('SAEXIS')->insert([
                'CodProd' => $codProd,
                'CodUbic' => $codUbic,
                'Existen' => -$cantidad // Existencia negativa, indica que se vendió sin stock
            ]);
        }
    }
    /**
     * Obtener ID de usuario desde token
     */
    private function getUserId($accessToken)
    {
        $urlUser = "https://api.mercadolibre.com/users/me";
        $ch = curl_init($urlUser);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $accessToken
        ]);

        $responseUser = curl_exec($ch);
        $userData = json_decode($responseUser, true);
        curl_close($ch);
        
        if (!isset($userData['id'])) {
            throw new \Exception("No se pudo obtener la información del usuario");
        }
        
        return $userData['id'];
    }
    
    /**
     * Obtener método de pago
     */
    private function getPaymentMethod($order)
    {
        if (isset($order['payments']) && !empty($order['payments'])) {
            $payment = $order['payments'][0];
            return $payment['payment_method_id'] ?? 'Desconocido';
        }
        return 'Desconocido';
    }
    
    /**
     * Refrescar token de acceso
     */
    private function refreshToken()
    {
        $clientId = $this->configModel->getConfig('ml_client_id')['valor'];
        $clientSecret = $this->configModel->getConfig('ml_client_secret')['valor'];
        $refreshToken = $this->configModel->getConfig('ml_refresh_token')['valor'];

        $ch = curl_init('https://api.mercadolibre.com/oauth/token');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
            'grant_type' => 'refresh_token',
            'client_id' => $clientId,
            'client_secret' => $clientSecret,
            'refresh_token' => $refreshToken
        ]));

        $response = curl_exec($ch);
        curl_close($ch);

        $data = json_decode($response, true);
        if (isset($data['access_token'])) {
            $this->configModel->setConfig('ml_access_token', $data['access_token']);
            $this->configModel->setConfig('ml_refresh_token', $data['refresh_token']);
            return $data['access_token'];
        }

        throw new \Exception('Error al refrescar el token');
    }
}