<?php

namespace App\Controllers;

class Jumpseller extends BaseController
{
    protected $configModel;
    protected $jumpsellerModel;
    protected $productosModel;

    public function __construct()
    {
        $this->configModel = new \App\Models\ConfigModel();
        $this->jumpsellerModel = new \App\Models\JumpsellerModel();
        $this->productosModel = new \App\Models\ProductosModel();
    }

    public function index()
    {
        $data['js_config'] = [
            'login' => $this->configModel->getConfig('js_login'),
            'auth_token' => $this->configModel->getConfig('js_auth_token'),
            'sync_active' => $this->configModel->getConfig('js_sync_active')
        ];

        return view('jumpseller/index', $data);
    }

    public function listado()
    {
        return view('jumpseller/listado');
    }

    public function toggleSync()
    {
        if ($this->request->isAJAX()) {
            $estado = $this->request->getPost('estado');
            $this->configModel->setConfig('js_sync_active', $estado);

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Estado de sincronización actualizado'
            ]);
        }
    }

    public function listarProductosJS()
    {
        try {
            $login = $this->configModel->getConfig('js_login')['valor'];
            $authToken = $this->configModel->getConfig('js_auth_token')['valor'];
            
            $allProducts = [];
            $page = 1;
            $limit = 100;
            
            do {
                $url = "https://api.jumpseller.com/v1/products.json?login={$login}&authtoken={$authToken}&limit={$limit}&page={$page}";
                
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json'
                ]);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                
                if ($httpCode !== 200) {
                    throw new \Exception("Error al obtener productos de Jumpseller. Código: {$httpCode}");
                }
                
                $data = json_decode($response, true);
                //var_dump('---');

                //var_dump(count($data));
                
                
                if (!is_array($data)) {
                    break;
                }
                
                $processedProducts = $this->processJumpsellerProducts($data);
                $allProducts = array_merge($allProducts, $processedProducts);
                //var_dump(count($allProducts));
                //var_dump('---');
                
                $page++;
                
                // Evitar bucle infinito
                if (count($data) < $limit) {
                    break;
                }
                
                usleep(100000); // 100ms delay para evitar rate limiting
                
            } while (count($data) == $limit);
            
            curl_close($ch);

            $allProductsnosku = $this->jumpsellerModel->syncJumpsellerData($allProducts);
            
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Productos obtenidos correctamente',
                'totalCargados' => count($allProducts),
                'TotalnoSku' => count($allProductsnosku),
                'noSku' => $allProductsnosku,
                'products' => $allProducts
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error al obtener productos: ' . $e->getMessage()
            ]);
        }
    }
    public function listarProductosJSb()
    {
        try {
            $allProducts = $this->jumpsellerModel->syncJumpsellerData();
            
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Productos obtenidos correctamente',
                'total' => count($allProducts),
                //'data' => $data,
                'products' => $allProducts
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Error al obtener productos: ' . $e->getMessage()
            ]);
        }
    }
    private function processJumpsellerProducts($products)
    {
        $processedProducts = [];
        
        foreach ($products as $product) {
            if (isset($product['product']['id'])) {
                $processedProducts[] = [
                    'id' => $product['product']['id'],
                    'name' => $product['product']['name'] ?? '',
                    'sku' => $product['product']['sku'] ?? '',
                    //'price' => $product['product']['price'] ?? 0,
                    'stock' => $product['product']['stock'] ?? 0,
                    'status' => $product['product']['status'] ?? '',
                    //'permalink' => $product['product']['permalink'] ?? ''
                ];
            }
        }
        
        return $processedProducts;
    }

    public function productos()
    {
        if ($this->request->isAJAX()) {
            $start = $this->request->getGet('start') ?? 0;
            $length = $this->request->getGet('length') ?? 10;
            $search = $this->request->getGet('search')['value'] ?? '';
            $order = $this->request->getGet('order')[0] ?? null;

            $columns = ['CodProd', 'Descrip', 'Existen', 'js_id', 'ultimo_sync'];
            $orderColumn = $columns[$order['column'] ?? 0] ?? 'CodProd';
            $orderDir = $order['dir'] ?? 'ASC';

            $result = $this->jumpsellerModel->getProductosConExistenciaPaginado(
                $start,
                $length,
                $search,
                $orderColumn,
                $orderDir
            );

            return $this->response->setJSON([
                'draw' => $this->request->getGet('draw'),
                'recordsTotal' => $result['recordsTotal'],
                'recordsFiltered' => $result['recordsFiltered'],
                'data' => $result['data']
            ]);
        }
    }

    public function sincronizarProducto()
    {
        if ($this->request->isAJAX()) {
            $jsId = $this->request->getPost('js_id');

            try {
                $login = $this->configModel->getConfig('js_login')['valor'];
                $authToken = $this->configModel->getConfig('js_auth_token')['valor'];
                
                $syncProduct = $this->jumpsellerModel->where('js_id', $jsId)->first();
                
                if (!$syncProduct) {
                    throw new \Exception("Producto no encontrado en sincronización");
                }

                $producto = $this->productosModel->getProductoConExistencia($syncProduct['cod_prod']);
                if (!$producto) {
                    throw new \Exception("Producto no encontrado en inventario");
                }

                $resultado = $this->actualizarStockJS($producto, $login, $authToken, $jsId);
                
                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Producto sincronizado correctamente',
                    'data' => $resultado
                ]);
                
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }
    }

    public function sincronizar()
    {
        if (!$this->configModel->getConfig('js_sync_active')['valor']) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'Sincronización desactivada'
            ]);
        }

        try {
            $login = $this->configModel->getConfig('js_login')['valor'];
            $authToken = $this->configModel->getConfig('js_auth_token')['valor'];
            $productos = $this->request->getPost('productos');

            // if (empty($selectedJsIds)) {
            //     return $this->response->setJSON([
            //         'success' => false,
            //         'message' => 'No hay elementos seleccionados'
            //     ]);
            // }

            if (empty($productos)) {
                $db = \Config\Database::connect();
                $productos = $db->table('js_productos_sync js')
                    ->select('js.js_id')
                    ->join('SAPROD p', 'js.cod_prod = p.Descrip2')
                    ->join('SAPROD_02 p2', 'js.cod_prod = p2.Descrip2')
                    ->where('p2.SINCRONIZACION', 1)
                    ->get()
                    ->getResultArray();
                
                $productos = array_column($productos, 'js_id');
            }

            $results = [];
            foreach ($productos as $jsId) {
                $syncProduct = $this->jumpsellerModel->where('js_id', $jsId)->first();
                if ($syncProduct) {
                    $producto = $this->productosModel->getProductoConExistencia($syncProduct['cod_prod']);
                    if ($producto) {
                        $result = $this->actualizarStockJS($producto, $login, $authToken, $jsId);
                        $results[] = $result;
                    }
                }
            }

            return $this->response->setJSON([
                'success' => true,
                'message' => 'Sincronización completada',
                'results' => $results
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function toggleProductSelection()
    {
        if ($this->request->isAJAX()) {
            $jsId = $this->request->getPost('js_id');
            $selected = $this->request->getPost('selected');
            
            try {
                $this->jumpsellerModel->toggleSelected($jsId, $selected);
                return $this->response->setJSON([
                    'success' => true,
                    'message' => 'Selección actualizada'
                ]);
            } catch (\Exception $e) {
                return $this->response->setJSON([
                    'success' => false,
                    'message' => $e->getMessage()
                ]);
            }
        }
        
        return $this->response->setJSON([
            'success' => false,
            'message' => 'Petición no válida'
        ]);
    }

    private function actualizarStockJS($producto, $login, $authToken, $jsId)
    {
        try {
            // Obtener el producto actual de Jumpseller
            $url = "https://api.jumpseller.com/v1/products/{$jsId}.json?login={$login}&authtoken={$authToken}";
            
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if ($httpCode !== 200) {
                throw new \Exception("Error al obtener producto de Jumpseller. Código: {$httpCode}");
            }
            
            $productData = json_decode($response, true);
            
            // Preparar datos para actualización
            $updateData = [
                'product' => [
                    'stock' => (int) $producto['Existen']
                ]
            ];
            
            // Actualizar el producto
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($updateData));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);
            
            $updateResponse = curl_exec($ch);
            $updateHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($updateHttpCode !== 200) {
                throw new \Exception("Error al actualizar stock en Jumpseller. Código: {$updateHttpCode}");
            }
            
            $updateResult = json_decode($updateResponse, true);
            
            // Registrar sincronización exitosa
            $this->jumpsellerModel->registrarSincronizacion(
                $producto['CodProd'],
                $jsId,
                1,
                (int) $producto['Existen']
            );
            
            return [
                'success' => true,
                'message' => 'Stock actualizado correctamente',
                'data' => [
                    'js_id' => $jsId,
                    'sku' => $producto['CodProd'],
                    'name' => $productData['name'] ?? '',
                    'new_stock' => (int) $producto['Existen'],
                    'previous_stock' => $productData['stock'] ?? 0
                ]
            ];
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
 * Sincroniza un producto por JS ID sin restricción AJAX
 * Para ser llamado desde scripts externos (Python, etc.)
 */
    public function sincronizarProductoInterno()
    {
        $jsId = $this->request->getPost('js_id') ?? $this->request->getGet('js_id');
        
        if (!$jsId) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'js_id es requerido'
            ]);
        }

        try {
            $login = $this->configModel->getConfig('js_login')['valor'];
            $authToken = $this->configModel->getConfig('js_auth_token')['valor'];
            
            // Verificar que la sincronización esté activa
            $syncActive = $this->configModel->getConfig('js_sync_active')['valor'];
            if (!$syncActive) {
                throw new \Exception("Sincronización desactivada");
            }
            
            $syncProduct = $this->jumpsellerModel->where('js_id', $jsId)->first();
            
            if (!$syncProduct) {
                throw new \Exception("Producto no encontrado en sincronización");
            }

            $producto = $this->productosModel->getProductoConExistencia($syncProduct['cod_prod']);
            if (!$producto) {
                throw new \Exception("Producto no encontrado en inventario");
            }

            $resultado = $this->actualizarStockJS($producto, $login, $authToken, $jsId);
            
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Producto sincronizado correctamente',
                'data' => $resultado,
                'cod_prod' => $syncProduct['cod_prod'],
                'stock_actualizado' => $producto['Existen']
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => $e->getMessage(),
                'js_id' => $jsId
            ]);
        }
    }

    /**
 * Obtiene elementos pendientes de la cola de sincronización
 * Para ser usado por scripts externos
 */
    public function obtenerCola()
    {
        $limite = $this->request->getGet('limite') ?? 10;
        
        try {
            $db = \Config\Database::connect();
            
            $items = $db->table('js_sync_queue')
                ->select('id, cod_prod, js_id, nuevo_stock, stock_anterior, tipo_movimiento, numero_documento, fecha_creacion, intentos')
                ->where('procesado', 0)
                ->where('intentos <', 5)
                ->orderBy('fecha_creacion', 'ASC')
                ->limit($limite)
                ->get()
                ->getResultArray();
            
            return $this->response->setJSON([
                'success' => true,
                'items' => $items,
                'total' => count($items)
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Marca un elemento de la cola como procesado
     */
    public function marcarProcesado()
    {
        $id = $this->request->getPost('id');
        $exito = $this->request->getPost('exito') ?? 1;
        $errorMensaje = $this->request->getPost('error_mensaje') ?? null;
        
        if (!$id) {
            return $this->response->setJSON([
                'success' => false,
                'message' => 'ID es requerido'
            ]);
        }

        try {
            $db = \Config\Database::connect();
            
            if ($exito) {
                // Marcar como procesado exitosamente
                $updateData = [
                    'procesado' => 1,
                    'fecha_procesado' => date('Y-m-d H:i:s')
                ];
                
                $db->table('js_sync_queue')
                    ->where('id', $id)
                    ->update($updateData);
            } else {
                // Incrementar intentos y guardar error
                $updateData = [
                    'error_mensaje' => $errorMensaje
                ];
                
                $db->table('js_sync_queue')
                    ->where('id', $id)
                    ->set('intentos', 'intentos + 1', false)
                    ->update($updateData);
            }
            
            return $this->response->setJSON([
                'success' => true,
                'message' => 'Estado actualizado correctamente'
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Obtiene estadísticas de la cola
     */
    public function estadisticasCola()
    {
        try {
            $db = \Config\Database::connect();
            
            // Para obtener estadísticas complejas, usamos una consulta más simple con múltiples queries
            $fechaLimite = date('Y-m-d H:i:s', strtotime('-1 day'));
            
            // Total de registros del último día
            $total = $db->table('js_sync_queue')
                ->where('fecha_creacion >=', $fechaLimite)
                ->countAllResults();
            
            // Pendientes
            $pendientes = $db->table('js_sync_queue')
                ->where('fecha_creacion >=', $fechaLimite)
                ->where('procesado', 0)
                ->countAllResults();
            
            // Procesados
            $procesados = $db->table('js_sync_queue')
                ->where('fecha_creacion >=', $fechaLimite)
                ->where('procesado', 1)
                ->countAllResults();
            
            // Con errores
            $conErrores = $db->table('js_sync_queue')
                ->where('fecha_creacion >=', $fechaLimite)
                ->where('procesado', 0)
                ->where('intentos >=', 3)
                ->countAllResults();
            
            // Más antiguo pendiente
            $masAntiguoPendiente = $db->table('js_sync_queue')
                ->select('fecha_creacion')
                ->where('procesado', 0)
                ->orderBy('fecha_creacion', 'ASC')
                ->limit(1)
                ->get()
                ->getRowArray();
            
            $stats = [
                'total' => $total,
                'pendientes' => $pendientes,
                'procesados' => $procesados,
                'con_errores' => $conErrores,
                'mas_antiguo_pendiente' => $masAntiguoPendiente['fecha_creacion'] ?? null
            ];
            
            return $this->response->setJSON([
                'success' => true,
                'estadisticas' => $stats
            ]);
            
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }
}